<?php
/*
Plugin Name: CoppermineSC Reloaded
Plugin URI: https://feelinalive.net/CoppermineSC-Reloaded
Description: Embed images from Coppermine Galleries into WordPress posts via shortcodes. Based on original script by <a href="http://www.lennartgroetzbach.de">Lennart Groetzbach</a> that was enhanced and adapted as a plugin by <a href="http://www.texx.org">Matthias Jell</a>. Further expanded upon by <a href="https://feelinalive.net">Feelin Alive Desgins</a> Compatible with PHP 8.0. 
Version: 1.1.1
Author: Feelin Alive Designs
Author URI: https://feelinalive.net
License: GPL2
*/
if (is_admin()) {
    require_once plugin_dir_path(__FILE__) . 'admin-settings.php';
}

if (!defined('ABSPATH')) {
    exit;
}

// Configuration
$cpg_path = $_SERVER['DOCUMENT_ROOT'] . get_option('coppermine_gallery_folder');
$cpg_url =  get_option('coppermine_gallery_albums_folder');
$cpg_home_url = get_option('coppermine_gallery_home');
define('CPGSC_GALLERY_HOME_URL', get_option('coppermine_gallery_home'));
define('CPGSC_GALLERY_URL', get_option('coppermine_gallery_domain') . get_option('coppermine_gallery_albums_folder')); 
define('CPG_TABLE_PREFIX', get_option('coppermine_table_prefix'));
define('CPG_GALLERY_FOLDERS', get_option('coppermine_gallery_albums_folder'));

if (!defined('CPGSC_GALLERY_PATH')) {
    define('CPGSC_GALLERY_PATH', get_option('coppermine_gallery_domain') . get_option('coppermine_gallery_folder'));
}




// Database connection
function cpgsc_get_connection() {
$host = esc_attr(get_option('coppermine_db_host', 'localhost'));
$user = get_option('coppermine_db_user');
$pass = get_option('coppermine_db_pass');
$dbname = get_option('coppermine_db_name');
    $mysqli = new mysqli($host, $user, $pass, $dbname);
    return $mysqli->connect_error ? false : $mysqli;
    
    
    
}

// Basic image shortcodes


// Optional: keep standard shortcode parsing
add_shortcode('cpg_image', fn($atts) => cpgsc_get_image_by_pid($atts, 'fullsize'));
add_shortcode('cpg_normal', fn($atts) => cpgsc_get_image_by_pid($atts, 'normal'));
add_shortcode('cpg_thumb', fn($atts) => cpgsc_get_image_by_pid($atts, 'thumb'));



function cpgsc_get_image_by_pid($atts, $size) {
    global $cpg_url, $cpg_home_url; 

    $pid = !empty($atts[0]) ? intval($atts[0]) : (!empty($atts['pid']) ? intval($atts['pid']) : 0);
    if (!$pid) return '<!-- Missing PID -->';

    $conn = cpgsc_get_connection();
    if (!$conn) return '<!-- DB connection failed -->';

    // Get filepath, filename, and album ID
    $stmt = $conn->prepare("SELECT filepath, filename, aid FROM " . CPG_TABLE_PREFIX . "pictures WHERE pid = ?");
    if (!$stmt) { error_log('Prepare failed: '.$conn->error); return '<!-- SQL prepare failed -->'; }
    $stmt->bind_param("i", $pid);
    $stmt->execute();
    $stmt->bind_result($filepath, $filename, $aid);
    $stmt->fetch();
    $stmt->close();

    if (!$filepath || !$filename) return '<!-- Picture not found -->';

    // Determine prefix for size
    $prefix = match($size) {
        'thumb'  => 'thumb_',
        'normal' => 'normal_',
        default  => '',
    };

    // Encode path
    $segments = array_map('rawurlencode', explode('/', trim($filepath, '/')));
    $encoded_path = implode('/', $segments);

    // Image URL
    $image_url = rtrim($cpg_url, '/') . '/' . $encoded_path . '/' . $prefix . rawurlencode($filename);

    // Display page URL
    $display_url = rtrim($cpg_home_url, '/') . '/displayimage.php?album=' . intval($aid) . '&pid=' . intval($pid) . '#top_display_media';

    return '<a href="' . esc_url($display_url) . '" target="_blank">'
         . '<img src="' . esc_url($image_url) . '" alt="' . esc_attr($filename) . '" />'
         . '</a>';
}

function cpgsc_get_album_by_pid($pid) {
    $conn = cpgsc_get_connection();
    if (!$conn) {
        return false; // Could not connect
    }

    $pid = intval($pid);
    $sql = "SELECT aid FROM " . CPG_TABLE_PREFIX . "pictures WHERE pid = {$pid} LIMIT 1";
    $res = $conn->query($sql);

    if (!$res || $res->num_rows === 0) {
        $conn->close();
        return false; // Image not found
    }

    $row = $res->fetch_assoc();
    $album_id = intval($row['aid']);

    // Now get album info
    $sql2 = "SELECT title FROM " . CPG_TABLE_PREFIX . "albums WHERE aid = {$album_id} LIMIT 1";
    $res2 = $conn->query($sql2);

    if (!$res2 || $res2->num_rows === 0) {
        $conn->close();
        return false; // Album not found
    }

    $album = $res2->fetch_assoc();
    $album_title = htmlspecialchars_decode($album['title'], ENT_QUOTES);
    $album_link  = cpgsc_get_album_url($album_id);

    $conn->close();

    return [
        'title' => $album_title,
        'link'  => $album_link
    ];
}
 




add_filter('the_content', function($content) {
    return preg_replace_callback(
        '/\[cpg_(image|normal|thumb)(?::(\d+))?\]/',
        function($matches) {
            $type = $matches[1];          // 'image', 'normal', 'thumb'
            $pid  = isset($matches[2]) ? intval($matches[2]) : 0;

            // Map type to size
            $size = match($type) {
                'thumb' => 'thumb',
                'normal' => 'normal',
                default => 'fullsize',
            };

            if (!$pid) return '<!-- Missing PID -->';

            return cpgsc_get_image_by_pid(['pid' => $pid], $size);
        },
        $content
    );
});

// Random thumbnails shortcode
add_filter('the_content', function($content) {
    return preg_replace_callback(
        '/\[cpg_albumrand:(\d+)(?:,(\d+))?\]/',
        function ($m) {
            $album_id = (int) $m[1];
            $limit    = isset($m[2]) ? (int) $m[2] : 0; // 0 = unlimited

            return cpgsc_render_album_thumbs(
                $album_id,
                $limit,
                'RAND()',
                ''
            );
        },
        $content
    );
});


// Recent thumbnails shortcode
add_filter('the_content', function($content) {
    return preg_replace_callback(
        '/\[cpg_albumnew:(\d+)(?:,(\d+))?\]/',
        function ($m) {
            $album_id = (int) $m[1];
            $limit    = isset($m[2]) ? (int) $m[2] : 0; // 0 = unlimited

            return cpgsc_render_album_thumbs(
                $album_id,
                $limit,
                'pid DESC', // newest first
                ''
            );
        },
        $content
    );
});

add_shortcode( 'cpg_album', 'cpgsc_render_album_shortcode' );

/**
 * Shortcode handler for [cpg_album:ID]
 * Example: [cpg_album:4]
 */
function cpgsc_render_album_shortcode( $atts = [], $content = null, $shortcode_tag = '' ) {
    // Parse ID from shortcode like [cpg_album:4]
    $album_id = 0;
    if ( strpos( $shortcode_tag, ':' ) !== false ) {
        $parts = explode( ':', $shortcode_tag );
        if ( isset( $parts[1] ) ) {
            $album_id = intval( $parts[1] );
        }
    }

    if ( ! $album_id ) {
        return '<!-- No album ID provided -->';
    }

    // No limit = pass a huge number (or remove LIMIT in SQL if you want true "all")
    $unlimited = PHP_INT_MAX;

    return cpgsc_render_album_thumbs(
        $album_id,
        $unlimited,
        'pid DESC', // default order, newest first
        ''          // no custom label
    );
}




function cpgsc_render_album_thumbs($album, $count, $order, $label) {
    global $cpg_url, $cpg_home_url;

    $conn = cpgsc_get_connection();
    if (!$conn) return '<!-- DB connection failed -->';

    // Get album title
    $stmt = $conn->prepare("SELECT title FROM " . CPG_TABLE_PREFIX . "albums WHERE aid = ?");
    $stmt->bind_param("i", $album);
    $stmt->execute();
    $stmt->bind_result($title);
    $stmt->fetch();
    $stmt->close();

    if (!$title) return '<!-- Album not found -->';

    // Build SQL depending on count
    if ($count > 0) {
        $stmt = $conn->prepare("SELECT pid, filepath, filename FROM " . CPG_TABLE_PREFIX . "pictures WHERE aid = ? ORDER BY $order LIMIT ?");
        $stmt->bind_param("ii", $album, $count);
    } else {
        $stmt = $conn->prepare("SELECT pid, filepath, filename FROM " . CPG_TABLE_PREFIX . "pictures WHERE aid = ? ORDER BY $order");
        $stmt->bind_param("i", $album);
    }

    $stmt->execute();
    $stmt->bind_result($pid, $filepath, $filename);

    $out = '<div class="cpgsc-album-thumbs">';

    $show_title = get_option('coppermine_show_album_title', '1');
    if ($show_title === '1') {
        $display_label = $label ?: $title;
        $out .= '<span class="cpgsc-album-thumbs-title">' . esc_html($display_label) . '</span>';
    }

    while ($stmt->fetch()) {
        $thumb_url  = esc_url($cpg_url . $filepath . 'thumb_' . $filename);
        $image_link = esc_url($cpg_home_url . 'displayimage.php?album=' . $album . '&pid=' . $pid . '#top_display_media');
        $out .= '<a href="' . $image_link . '" target="_blank" class="cpgsc-album-thumb-link">';
        $out .= '<img src="' . $thumb_url . '" alt="' . esc_attr($filename) . '" />';
        $out .= '</a>';
    }

    $stmt->close();
    return $out . '</div>';
}


// Helper function to get all subcategory IDs recursively
function cpgsc_get_subcategories($parent_id, $conn) {
    $categories = [$parent_id];
    $sql = "SELECT cid FROM " . CPG_TABLE_PREFIX . "categories WHERE parent = " . intval($parent_id);
    $result = $conn->query($sql);

    if ($result) {
        while ($row = $result->fetch_assoc()) {
            $sub_id = (int)$row['cid'];
            $categories = array_merge($categories, cpgsc_get_subcategories($sub_id, $conn));
        }
        $result->free();
    }

    return $categories;
}

// Fetch recent albums based on latest picture upload date, including subcategories
function cpgsc_get_recent_albums($category_id = null, $limit = null) {
    $conn = cpgsc_get_connection();
    if (!$conn) {
        return [];
    }

    // Determine which categories to include
    $category_filter = '';
    if ($category_id !== null) {
        $category_ids = cpgsc_get_subcategories((int)$category_id, $conn);
        $category_list = implode(',', array_map('intval', $category_ids));
        $category_filter = "WHERE a.category IN ($category_list)";
    }

    $sql = "
        SELECT a.aid, a.title, a.category,
               COUNT(p.pid) AS image_count,
               MAX(p.ctime) AS last_update,
               (SELECT CONCAT(p2.filepath, 'thumb_', p2.filename)
                FROM " . CPG_TABLE_PREFIX . "pictures AS p2
                WHERE p2.aid = a.aid
                ORDER BY p2.ctime DESC
                LIMIT 1) AS thumb_path
        FROM " . CPG_TABLE_PREFIX . "albums AS a
        LEFT JOIN " . CPG_TABLE_PREFIX . "pictures AS p ON a.aid = p.aid
        $category_filter
        GROUP BY a.aid
        HAVING last_update IS NOT NULL
        ORDER BY last_update DESC";

    // Add LIMIT if set
    if ($limit !== null && is_numeric($limit)) {
        $sql .= " LIMIT " . intval($limit);
    }

    $albums = [];
    $result = $conn->query($sql);

    if ($result) {
        while ($row = $result->fetch_assoc()) {
            $albums[] = $row;
        }
        $result->free();
    }

    $conn->close();
    return $albums;
}



// Display recent albums globally or in a category
function cpgsc_display_recent_albums($category_id = null, $limit = null) {
    $albums = cpgsc_get_recent_albums($category_id, $limit);
    if (empty($albums)) {
        return '<p>No recent albums found.</p>';
    }

    $output = '<div class="cpgsc-recent-albums">';
    foreach ($albums as $album) {
        $url = CPGSC_GALLERY_PATH . '/thumbnails.php?album=' . $album['aid'];
        
        $title = htmlspecialchars_decode($album['title']);
        $image_count = intval($album['image_count']);
        $last_update = date("F j, Y", intval($album['last_update']));

        $thumb_rel_path = $album['thumb_path'];
        $full_rel_path = str_replace('thumb_', '', $thumb_rel_path);
$thumb_file = $_SERVER['DOCUMENT_ROOT'] . CPG_GALLERY_FOLDERS . $thumb_rel_path;
$fallback_file = $_SERVER['DOCUMENT_ROOT'] . CPG_GALLERY_FOLDERS . $full_rel_path;

if (file_exists($thumb_file)) {
    $thumb_url = CPGSC_GALLERY_URL . rawurlencode($thumb_rel_path);
} elseif (file_exists($fallback_file)) {
    $thumb_url = CPGSC_GALLERY_URL . rawurlencode($full_rel_path);
} else {
    $thumb_url = '';
}


$img_tag = $thumb_url
    ? "<img src='$thumb_url' alt='$title' style='display: block; text-algin:center;'>"
    : '<div style="width:150px; height:100px; background:#ccc; display:flex; align-items:center; justify-content:center;">No Image</div>';


        $output .= "<div class='cpgsc-recent-album'>
                        <a href='$url'>$img_tag</a>
                        <div class='cpgsc-recent-albums-title'><a href='$url'><strong>$title</strong></a></div>
                        <div class='cpgsc-recent-albums-info'>$image_count images<br>Updated: $last_update</div>
                    </div>";
    }
    $output .= '</div>';
    return $output;
	
}



// Shortcode: [cpg_cat:CATID] – display all images in a category
add_filter('the_content', function($content) {
    return preg_replace_callback(
        '/\[cpg_cat:(\d+)(?:,(\d+))?\]/',
        function($matches) {
            $cat_id = intval($matches[1]);
            $limit = isset($matches[2]) ? intval($matches[2]) : 0; // 0 = unlimited
            return cpgsc_render_category_images($cat_id, $limit, 'pid ASC'); // normal order
        },
        $content
    );
});

// Shortcode: [cpg_catrand:CATID] – random images in a category
add_filter('the_content', function($content) {
    return preg_replace_callback(
        '/\[cpg_catrand:(\d+)(?:,(\d+))?\]/',
        function($matches) {
            $cat_id = intval($matches[1]);
            $limit = isset($matches[2]) ? intval($matches[2]) : 0; 
            return cpgsc_render_category_images($cat_id, $limit, 'RAND()'); // random order
        },
        $content
    );
});

// Shortcode: [cpg_catnew:CATID] – newest images in a category
add_filter('the_content', function($content) {
    return preg_replace_callback(
        '/\[cpg_catnew:(\d+)(?:,(\d+))?\]/',
        function($matches) {
            $cat_id = intval($matches[1]);
            $limit = isset($matches[2]) ? intval($matches[2]) : 0; 
            return cpgsc_render_category_images($cat_id, $limit, 'pid DESC'); // newest first
        },
        $content
    );
});

// Core function to render images in a category
function cpgsc_render_category_images($category_id, $limit = 0, $order = 'pid ASC') {
    global $cpg_url, $cpg_home_url;

    $conn = cpgsc_get_connection();
    if (!$conn) return '<!-- DB connection failed -->';

    // Get all subcategories
    $category_ids = cpgsc_get_subcategories($category_id, $conn);
    $category_list = implode(',', array_map('intval', $category_ids));

    // Build query (fetch pid + aid for proper linking)
    $sql = "SELECT p.pid, p.aid, p.filepath, p.filename 
            FROM " . CPG_TABLE_PREFIX . "pictures AS p
            INNER JOIN " . CPG_TABLE_PREFIX . "albums AS a ON p.aid = a.aid
            WHERE a.category IN ($category_list)
            ORDER BY $order";

    if ($limit > 0) {
        $sql .= " LIMIT " . intval($limit);
    }

    $res = $conn->query($sql);
    if (!$res || $res->num_rows === 0) {
        $conn->close();
        return '<!-- No images found in this category -->';
    }

    $output = '<div class="cpgsc-category-images">';
    while ($row = $res->fetch_assoc()) {
        $thumb_url   = rtrim($cpg_url, '/') . '/' . trim($row['filepath'], '/') . '/thumb_' . rawurlencode($row['filename']);
        $display_url = rtrim($cpg_home_url, '/') . '/displayimage.php?album=' . intval($row['aid']) . '&pid=' . intval($row['pid']) . '#top_display_media';

        $output .= '<a href="' . esc_url($display_url) . '" target="_blank">';
        $output .= '<img src="' . esc_url($thumb_url) . '" alt="' . esc_attr($row['filename']) . '" />';
        $output .= '</a>';
    }
    $output .= '</div>';

    $conn->close();
    return $output;
}

// Shortcode: [cpg_galrand:COUNT] – random images from the whole gallery
add_filter('the_content', function($content) {
    return preg_replace_callback(
        '/\[cpg_galrand:(\d+)\]/',
        function($matches) {
            $limit = intval($matches[1]);
            return cpgsc_render_gallery_random($limit);
        },
        $content
    );
});

function cpgsc_render_gallery_random($limit = null) {
    global $cpg_url, $cpg_home_url;

    $conn = cpgsc_get_connection();
    if (!$conn) return '<!-- DB connection failed -->';

   $sql = "SELECT p.pid, p.filepath, p.filename, p.aid
        FROM " . CPG_TABLE_PREFIX . "pictures AS p
        WHERE approved = 'YES'
        ORDER BY RAND()";


    if (!empty($limit) && is_numeric($limit)) {
        $sql .= " LIMIT " . intval($limit);
    }

    $res = $conn->query($sql);
    if (!$res || $res->num_rows === 0) {
        $conn->close();
        return '<!-- No images found in gallery -->';
    }

    $output = '<div class="cpgsc-gallery-rand">';
   while ($row = $res->fetch_assoc()) {
    $thumb_url   = rtrim($cpg_url, '/') . '/' . trim($row['filepath'], '/') . '/thumb_' . rawurlencode($row['filename']);
    $display_url = rtrim($cpg_home_url, '/') . '/displayimage.php?album=' . intval($row['aid']) . '&pid=' . intval($row['pid']) . '#top_display_media';

    $output .= '<a href="' . esc_url($display_url) . '" target="_blank">';
    $output .= '<img src="' . esc_url($thumb_url) . '" alt="' . esc_attr($row['filename']) . '" />';
    $output .= '</a>';
}
    $output .= '</div>';

    $conn->close();
    return $output;
}






function cpgsc_get_gallery_stats() {
    $conn = cpgsc_get_connection();
    if (!$conn) {
        return '<p>Could not connect to the database.</p>';
    }

    $stats = [];

    // Total images
    $res = $conn->query("SELECT COUNT(*) AS count FROM " . CPG_TABLE_PREFIX . "pictures");
    $row = $res->fetch_assoc();
    $stats['images'] = $row['count'];

    // Total albums
    $res = $conn->query("SELECT COUNT(*) AS count FROM " . CPG_TABLE_PREFIX . "albums");
    $row = $res->fetch_assoc();
    $stats['albums'] = $row['count'];

    // Total categories
    $res = $conn->query("SELECT COUNT(*) AS count FROM " . CPG_TABLE_PREFIX . "categories");
    $row = $res->fetch_assoc();
    $stats['categories'] = $row['count'];

    // Total views
    $res = $conn->query("SELECT SUM(hits) AS total_views FROM " . CPG_TABLE_PREFIX . "pictures");
    $row = $res->fetch_assoc();
    $stats['total_views'] = $row['total_views'] ? $row['total_views'] : 0;

    // Most recent upload
    $res = $conn->query("SELECT MAX(ctime) AS last_upload FROM " . CPG_TABLE_PREFIX . "pictures");
    $row = $res->fetch_assoc();
    $stats['last_upload'] = $row['last_upload'] ? date("F j, Y", $row['last_upload']) : 'N/A';

    $conn->close();

    $output = "<div class='cpgsc-gallery-stats' style='font-size: 1em; line-height: 1.6;'>";
    $output .= "<strong>Gallery Statistics:</strong><br>";
    $output .= "Albums: " . $stats['albums'] . "<br>";
    $output .= "Images: " . $stats['images'] . "<br>";
    $output .= "Categories: " . $stats['categories'] . "<br>";
    $output .= "Total Views: " . $stats['total_views'] . "<br>";
    $output .= "Last upload: " . $stats['last_upload'];
    $output .= "</div>";

    return $output;
}


add_shortcode('cpg_gallerystats', 'cpgsc_get_gallery_stats');

function cpgsc_get_gallery_stats_short() {
    $conn = cpgsc_get_connection();
    if (!$conn) {
        return '<p>Could not connect to the database.</p>';
    }

    $stats = [];

    // Total images
    $res = $conn->query("SELECT COUNT(*) AS count FROM " . CPG_TABLE_PREFIX . "pictures");
    $row = $res->fetch_assoc();
    $stats['images'] = $row['count'];

    // Total albums
    $res = $conn->query("SELECT COUNT(*) AS count FROM " . CPG_TABLE_PREFIX . "albums");
    $row = $res->fetch_assoc();
    $stats['albums'] = $row['count'];

    // Total categories
    $res = $conn->query("SELECT COUNT(*) AS count FROM " . CPG_TABLE_PREFIX . "categories");
    $row = $res->fetch_assoc();
    $stats['categories'] = $row['count'];

    // Total views
    $res = $conn->query("SELECT SUM(hits) AS total_views FROM " . CPG_TABLE_PREFIX . "pictures");
    $row = $res->fetch_assoc();
    $stats['total_views'] = $row['total_views'] ? $row['total_views'] : 0;

    // Most recent upload
    $res = $conn->query("SELECT MAX(ctime) AS last_upload FROM " . CPG_TABLE_PREFIX . "pictures");
    $row = $res->fetch_assoc();
    $stats['last_upload'] = $row['last_upload'] ? date("F j, Y", $row['last_upload']) : 'N/A';

    $conn->close();

    $output = "<div class='cpgsc-gallery-stats' style='font-size: 1em; line-height: 1.6;'>";
     $output .= "Gallery contains ";
    $output .= "<b>" . $stats['images'] . "</b> images ";
    $output .= "in <b>" . $stats['albums'] . "</b>albums ";
    $output .= "in <b>" . $stats['categories'] . "</b> categories ";
    $output .= "with <b>" . $stats['total_views'] . "</b> total views <br>";
    $output .= "Last Updated on: " . $stats['last_upload'];
    $output .= "</div>";
    
    

    return $output;
}


add_shortcode('cpg_gallerystatsshort', 'cpgsc_get_gallery_stats_short');


function cpgsc_latest_images_shortcode($atts) {
    $conn = cpgsc_get_connection();
    if (!$conn) {
        return '<p>Could not connect to the database.</p>';
    }

    // Parse [cpg_latestimages:6]
    $limit = isset($atts[0]) && is_numeric($atts[0]) ? intval($atts[0]) : null;

$sql = "SELECT pid, filepath, filename, title 
        FROM " . CPG_TABLE_PREFIX . "pictures 
        WHERE approved = 'YES' 
        ORDER BY ctime DESC";

if ($limit !== null) {
    $sql .= " LIMIT " . $limit;
}

    $res = $conn->query($sql);
    if (!$res) {
        return '<p>Error fetching latest images: ' . $conn->error . '</p>';
    }

    $output = "<div class='cpgsc-latest-images' style='display: block;'>";

    while ($row = $res->fetch_assoc()) {
        $thumb_url = cpgsc_get_image_url($row['filepath'], $row['filename'], 'thumb_');
        $display_url = rtrim(CPGSC_GALLERY_PATH, '/') . '/displayimage.php?pid=' . intval($row['pid']);
        $title       = htmlspecialchars_decode($row['title'] ?? '', ENT_QUOTES);

        $output .= "<div class='cpgsc-latestimgs-img' style='text-align: center; max-width: 120px;' >";
        $output .= "<a href='{$display_url}' target='_blank'>";
        $output .= "<img src='{$thumb_url}' style='width: 100%; height: auto;' alt='{$title}' />";
        $output .= "</a>";
        if ($title) {
            $output .= "<div style='font-size: 0.9em; margin-top: 5px;'>{$title}</div>";
        }
        $output .= "</div>";
    }

    $output .= "</div>";
    $conn->close();

    return $output;
    
    
}




function cpgsc_get_image_url($filepath, $filename, $thumb_prefix = '') {
    $filepath = trim($filepath, '/'); // removes leading/trailing slashes
    $filename = $thumb_prefix . $filename;

    return rtrim(CPGSC_GALLERY_PATH, '/') . '/albums/' . $filepath . '/' . $filename;
}



add_shortcode('cpg_latestimages', 'cpgsc_latest_images_wrapper');

function cpgsc_latest_images_wrapper($atts = [], $content = '', $tag = '') {
    // Get full shortcode tag, e.g. 'cpg_latestimages:5'
    global $shortcode_tags;

    // Find the actual shortcode tag used (e.g., 'cpg_latestimages:5')
    $called_shortcode = array_search('cpgsc_latest_images_wrapper', $shortcode_tags);

    // Extract number after colon
    $parts = explode(':', $called_shortcode);
    $count = isset($parts[1]) && is_numeric($parts[1]) ? intval($parts[1]) : null;

    return cpgsc_latest_images_render($count);
}

function cpgsc_latest_images_render($limit = null) {
    global $cpg_home_url;

    $conn = cpgsc_get_connection();
    if (!$conn) {
        return '<p>Could not connect to the database.</p>';
    }

    // Add p.hits to the select for image views
    $sql = "SELECT p.pid, p.filepath, p.filename, p.title AS image_title, p.hits, a.title AS album_title
            FROM " . CPG_TABLE_PREFIX . "pictures AS p
            INNER JOIN " . CPG_TABLE_PREFIX . "albums AS a ON p.aid = a.aid
            WHERE p.approved = 'YES'
            ORDER BY p.ctime DESC";

    if ($limit > 0) {
        $sql .= " LIMIT " . intval($limit);
    }

    $res = $conn->query($sql);
    if (!$res) {
        return '<p>Error fetching latest images: ' . $conn->error . '</p>';
    }

    // Get display mode from admin settings
   $display_mode = get_option('coppermine_display_tooltip', 'album');

if ($display_mode === 'album') {
    $title_attr = $album_title;
} elseif ($display_mode === 'views') {
    $title_attr = "{$views} views";
} else {
    $title_attr = $image_title;
}

    $output = "<div class='cpgsc-latest-images' style='display: block;'>";

    while ($row = $res->fetch_assoc()) {
        $thumb_url = cpgsc_get_image_url($row['filepath'], $row['filename'], 'thumb_');
        $display_url = $cpg_home_url . 'displayimage.php?pid=' . intval($row['pid']);
        
        $image_title = htmlspecialchars($row['image_title'] ?? '', ENT_QUOTES);
        $album_title = htmlspecialchars($row['album_title'] ?? '', ENT_QUOTES);
        $views = intval($row['hits']);

        // Choose title or views based on setting
        $title_attr = $display_mode = get_option('coppermine_display_tooltip', 'album');

if ($display_mode === 'album') {
    $title_attr = $album_title;
} elseif ($display_mode === 'views') {
    $title_attr = "{$views} views";
} else {
    $title_attr = $image_title;
}

        $output .= "<div class='cpgsc-latest-images-img' style='display: inline-block'>";
        $output .= "<a href='{$display_url}' target='_blank'>";
        $output .= "<img src='{$thumb_url}' alt='{$image_title}' title='{$title_attr}' />";
        $output .= "</a>";
        $output .= "</div>";
    }

    $output .= "</div>";
    $conn->close();

    return $output;
}




function cpgsc_render_album_breadcrumb($album_id) {
    $conn = cpgsc_get_connection();
    if (!$conn) {
        return '<p>Could not connect to the database.</p>';
    }

    $album_id = intval($album_id);
    $sql = "SELECT a.aid, a.title AS album_title, a.category, c.name AS category_name, c.cid, c.parent
            FROM " . CPG_TABLE_PREFIX . "albums AS a
            LEFT JOIN " . CPG_TABLE_PREFIX . "categories AS c ON a.category = c.cid
            WHERE a.aid = {$album_id}";

    $res = $conn->query($sql);
    if (!$res || $res->num_rows === 0) {
        return '<p>Album not found.</p>';
    }

    $row = $res->fetch_assoc();
    $album_title = htmlspecialchars_decode($row['album_title'], ENT_QUOTES);
    $album_link = cpgsc_get_album_url($row['aid']); // You need to define this function to match your URL structure

    // Build category hierarchy if needed
    $breadcrumbs = [];
    $cid = $row['cid'];
    while ($cid && $cid != 0) {
        $cat_sql = "SELECT cid, name, parent FROM " . CPG_TABLE_PREFIX . "categories WHERE cid = {$cid}";
        $cat_res = $conn->query($cat_sql);
        if ($cat_res && $cat_res->num_rows) {
            $cat = $cat_res->fetch_assoc();
            array_unshift($breadcrumbs, htmlspecialchars_decode($cat['name'], ENT_QUOTES));
            $cid = $cat['parent'];
        } else {
            break;
        }
    }

    $conn->close();

    $breadcrumbs_str = implode(' > ', $breadcrumbs);
    $breadcrumbs_str .= ' > <a href="' . $album_link . '">' . $album_title . '</a>';

    return '<div class="cpgsc-breadcrumb"> Home > '  . $breadcrumbs_str . '</div>';
}

function cpgsc_get_album_url($album_id) {
    return CPGSC_GALLERY_HOME_URL . '/thumbnails.php?album=' . intval($album_id);
}

add_filter('the_content', function($content) {
    return preg_replace_callback('/\[cpg_albumlink:(\d+)\]/', function ($matches) {
        $album_id = (int)$matches[1];
        return cpgsc_render_album_breadcrumb($album_id);
    }, $content);
});

add_filter('the_content', function($content) {
    return preg_replace_callback('/\[cpg_latestalbumsalltext:(\d+)\]/', function ($matches) {
        $limit = (int)$matches[1];
        return cpgsc_display_latest_albums_text($limit);
    }, $content);
});

add_shortcode('cpg_latestalbumstext', function ($atts) {
    $args = explode(',', $atts[0]);
    $category_id = isset($args[0]) ? intval($args[0]) : 1;
    $limit = isset($args[1]) ? intval($args[1]) : 5;
    return cpgsc_display_latest_albums_by_category_text($category_id, $limit);
});


function cpgsc_display_latest_albums_text($limit = null) {
    $conn = cpgsc_get_connection();
    if (!$conn) {
        return '<p>Could not connect to the Coppermine database.</p>';
    }

    $limit = intval($limit);

    $sql = "
        SELECT a.aid, a.title, MAX(p.ctime) AS last_upload
        FROM " . CPG_TABLE_PREFIX . "albums AS a
        JOIN " . CPG_TABLE_PREFIX . "pictures AS p ON a.aid = p.aid
        WHERE p.approved = 'YES'
        GROUP BY a.aid
        ORDER BY last_upload DESC
        LIMIT $limit
    ";

    $res = $conn->query($sql);
    if (!$res || $res->num_rows === 0) {
        $conn->close();
        return '<p>No albums with images found.</p>';
    }

    $output = '<div class="cpgsc-latest-albums-text"> <span class="cpgsc-latest-albums-text-title">Latest Albums in the Gallery</span>';
    while ($row = $res->fetch_assoc()) {
        $album_id = $row['aid'];
        $album_title = htmlspecialchars_decode($row['title'], ENT_QUOTES);
        $last_update = $row['last_upload'] ? date("F j, Y", $row['last_upload']) : 'Unknown Date';
        $album_url = CPGSC_GALLERY_PATH . '/thumbnails.php?album=' . $album_id;

        $output .= "<div class='cpgsc-latest-albums-text-link'>$last_update — <a href='$album_url'>$album_title</a></div>";
    }
    $output .= '</div>';

    $conn->close();
    return $output;
}
add_filter('the_content', function($content) {
    return preg_replace_callback('/\[cpg_latestalbumstext:(\d+),(\d+)\]/', function ($matches) {
        $category_id = (int)$matches[1];
        $limit = (int)$matches[2];
        return cpgsc_display_latest_albums_by_category_text($category_id, $limit);
    }, $content);
});

function cpgsc_display_latest_albums_by_category_text($category_id, $limit = null) {
    $conn = cpgsc_get_connection();
    if (!$conn) {
        return '<p>Could not connect to the Coppermine database.</p>';
    }

    $limit = intval($limit);
    $category_id = intval($category_id);

    // Get the main category name
    $cat_sql = "SELECT name FROM " . CPG_TABLE_PREFIX . "categories WHERE cid = $category_id";
    $cat_res = $conn->query($cat_sql);
    $category_title = ($cat_res && $cat_res->num_rows) ? htmlspecialchars_decode($cat_res->fetch_assoc()['name'], ENT_QUOTES) : 'Category';

    // Get all subcategory IDs recursively
    $category_ids = [$category_id];
    $checked = [];

    while (!empty($category_ids)) {
        $cid = array_shift($category_ids);
        $checked[] = $cid;

        $sub_sql = "SELECT cid FROM " . CPG_TABLE_PREFIX . "categories WHERE parent = $cid";
        $sub_res = $conn->query($sub_sql);
        if ($sub_res && $sub_res->num_rows > 0) {
            while ($sub = $sub_res->fetch_assoc()) {
                $category_ids[] = $sub['cid'];
            }
        }
    }

    $checked_ids = implode(',', array_map('intval', $checked)); // safe for SQL

    // Fetch latest albums
    $sql = "
        SELECT a.aid, a.title AS album_title, MAX(p.ctime) AS last_upload, c.name AS album_category
        FROM " . CPG_TABLE_PREFIX . "albums AS a
        JOIN " . CPG_TABLE_PREFIX . "pictures AS p ON a.aid = p.aid
        LEFT JOIN " . CPG_TABLE_PREFIX . "categories AS c ON a.category = c.cid
        WHERE a.category IN ($checked_ids) AND p.approved = 'YES'
        GROUP BY a.aid
        ORDER BY last_upload DESC
        LIMIT $limit
    ";

    $res = $conn->query($sql);
    if (!$res || $res->num_rows === 0) {
        $conn->close();
        return '<p>No albums found in this category.</p>';
    }

    // Output header
 $output = "<div class='cpgsc-latest-albums-category-text'> <span class='cpgsc-latest-albums-text-title'>";
    $output .= "Latest Albums in <b>$category_title</b></span>";

    while ($row = $res->fetch_assoc()) {
        $album_id = $row['aid'];
        $album_title = htmlspecialchars_decode($row['album_title'], ENT_QUOTES);
        $album_category = htmlspecialchars_decode($row['album_category'], ENT_QUOTES);
        $last_update = $row['last_upload'] ? date("F j, Y", $row['last_upload']) : 'Unknown Date';
        $album_url = CPGSC_GALLERY_PATH . '/thumbnails.php?album=' . $album_id;

        $output .= "<div class='cpgsc-latest-albums-text-link'>$album_category — <a href='$album_url'>$album_title</a> <span style='color: #888;'>($last_update)</span></div>";
    }

    $output .= '</div>';
    $conn->close();
    return $output;
}







// Regex-based shortcode handler for [cpg_latestalbums:49,8]
function cpgsc_parse_category_shortcode($tag) {
    $pattern = '/\[cpg_latestalbums:([0-9]+),([0-9]+)\]/';
    return preg_replace_callback($pattern, function($matches) {
        $category_id = intval($matches[1]);
        $limit = intval($matches[2]);
        return cpgsc_display_recent_albums($category_id, $limit);
    }, $tag);
}

// Regex-based shortcode handler for [cpg_latestalbumsall:8]
function cpgsc_parse_global_shortcode($tag) {
    $pattern = '/\[cpg_latestalbumsall:([0-9]+)\]/';
    return preg_replace_callback($pattern, function($matches) {
        $limit = intval($matches[1]);
        return cpgsc_display_recent_albums(null, $limit);
    }, $tag);
}

// Apply regex-based handlers to post content
function cpgsc_process_custom_shortcodes($content) {
    $content = cpgsc_parse_category_shortcode($content);
    $content = cpgsc_parse_global_shortcode($content);
    return $content;
}

add_filter( 'the_content', 'cpgsc_album_shortcode_handler' );

function cpgsc_album_shortcode_handler( $content ) {
    return preg_replace_callback(
        '/\[cpg_album:(\d+)(?:,(\d+))?\]/',
        'cpgsc_album_shortcode_replace',
        $content
    );
}


function cpgsc_album_shortcode_replace( $matches ) {
    $album_id = intval( $matches[1] );
    $limit    = isset($matches[2]) ? intval($matches[2]) : 0; // 0 = unlimited

    if ( ! $album_id ) {
        return '<!-- Invalid album ID -->';
    }

    return cpgsc_render_album_thumbs(
        $album_id,
        $limit,      // pass limit if provided, else 0 (unlimited)
        'pid DESC',  // default order
        ''           // no custom label
    );
}



add_filter('the_content', 'cpgsc_process_custom_shortcodes');

add_filter('the_content', 'cpgsc_parse_custom_colon_shortcodes');

function cpgsc_parse_custom_colon_shortcodes($content) {
    // [cpg_latestimages:6]
    $content = preg_replace_callback('/\[cpg_latestimages:(\d+)\]/', function ($matches) {
        $limit = intval($matches[1]);
        return cpgsc_latest_images_render($limit);
    }, $content);

    return $content;
}



function cpgsc_enqueue_styles() {
    wp_enqueue_style(
        'cpgsc-style', 
        plugin_dir_url(__FILE__) . 'css/style.css',
        array(),              // No dependencies
        '1.0'                 // Version
    );
}
add_action('wp_enqueue_scripts', 'cpgsc_enqueue_styles');









